/**
* \file: app_iface_dbus.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: authorization level daemon
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/

//contains the implementation of a dbus "application interface binding"

#include <glib.h>
#include <string.h>

#include "util/logger.h"
#include "encryption/signature.h"
#include "app_iface/app_iface.h"
#include "app_iface/generated/ald-dbus-interface.h"

#define ALD_BUS_NAME              	"com.adit.de.ALD"
#define ALD_LEVEL_STATUS_OBJECT      		"/com/adit/de/ALD/level_status"
#define ALD_CHANGE_LEVEL_OBJECT     "/com/adit/de/ALD/change_level"

static Level_status *level_status_iface=NULL;
static Change_level *change_level_iface=NULL;
static guint bus_name_owner_id=0;
static GDBusMethodInvocation *pending_request_invocation=NULL;

static void app_iface_dbus_on_connected(GDBusConnection *con, const gchar *name, gpointer data);
static void app_iface_dbus_on_name_aquired(GDBusConnection *con, const gchar* name, gpointer data);
static void app_iface_dbus_on_connection_lost(GDBusConnection *con, const gchar* name, gpointer data);

static gboolean app_iface_dbus_on_challenge_request(Change_level *iface,GDBusMethodInvocation *invocation,
		const gchar *arg_ecu_id);
static gboolean app_iface_dbus_on_response_request(Change_level *iface,GDBusMethodInvocation *invocation,
		GVariant *arg_response);
static gboolean app_iface_dbus_on_lock_device_request(Change_level *iface,GDBusMethodInvocation *invocation);

static void app_iface_dbus_level_change_request_callback(error_code_t result, bool did_level_change);
static void app_iface_dbus_on_lock_device_request_callback(error_code_t result, bool did_level_change);

//------------------------------ provides functions of the application interface binding -----------------------------
error_code_t app_iface_init(void)
{
	level_status_iface=level_status_skeleton_new();
	change_level_iface=change_level_skeleton_new();
	bus_name_owner_id = g_bus_own_name(G_BUS_TYPE_SYSTEM, ALD_BUS_NAME,G_BUS_NAME_OWNER_FLAGS_NONE,
								app_iface_dbus_on_connected,app_iface_dbus_on_name_aquired,
								app_iface_dbus_on_connection_lost, NULL,NULL);
	return RESULT_OK;
}

void app_iface_deinit(void)
{
	if (bus_name_owner_id!=0)
		g_bus_unown_name(bus_name_owner_id);

	if (level_status_iface!=NULL)
	{
		g_object_unref(level_status_iface);
		level_status_iface=NULL;
	}

	if (change_level_iface!=NULL)
	{
		g_object_unref(change_level_iface);
		change_level_iface=NULL;
	}
}

void app_iface_signal_level_changing(security_level_t targeted_level)
{
	if (level_status_iface==NULL)
		return;
	level_status_emit_new_level(level_status_iface, (guint)targeted_level);
}

void app_iface_signal_level_changed(security_level_t new_level)
{
	if (level_status_iface==NULL)
		return;

	level_status_set_level(level_status_iface,(guint)new_level);
	level_status_emit_level_change_done(level_status_iface, (guint)new_level);
}
//--------------------------------------------------------------------------------------------------------------------

//------------------------------ private functions -------------------------------------------------------------------
static void app_iface_dbus_on_connected(GDBusConnection *con, const gchar *name, gpointer data)
{
	(void)data;
	logger_log_debug("APP_IFACE_DBUS - Name %s aquired on system bus.",name);
	g_dbus_connection_set_exit_on_close(con, FALSE);

	(void) g_signal_connect(change_level_iface, "handle-challenge", G_CALLBACK(app_iface_dbus_on_challenge_request), NULL);
	(void) g_signal_connect(change_level_iface, "handle-response", G_CALLBACK(app_iface_dbus_on_response_request), NULL);
	(void) g_signal_connect(change_level_iface, "handle-lock-device", G_CALLBACK(app_iface_dbus_on_lock_device_request), NULL);

	/* PRQA: Lint Message 826: deactivation because this is generated code*/
	/*lint -save -e826*/
	if(g_dbus_interface_skeleton_export(G_DBUS_INTERFACE_SKELETON(level_status_iface),
                                      con,ALD_LEVEL_STATUS_OBJECT,NULL) != TRUE)
	{
		logger_log_error("Error exporting the Level_status interface to the system bus. Shutting down.");
		app_iface_request_daemon_exit_due_to_errors(RESULT_DBUS_SYSTEM_BUS_ERROR);
		return;
	}

	if(g_dbus_interface_skeleton_export(G_DBUS_INTERFACE_SKELETON(change_level_iface),
                                      con,ALD_CHANGE_LEVEL_OBJECT,NULL) != TRUE)
	{
		logger_log_error("Error exporting the Change_level interface to the system bus. Shutting down.");
		app_iface_request_daemon_exit_due_to_errors(RESULT_DBUS_SYSTEM_BUS_ERROR);
		return;
	}
	/*lint -restore*/
}

static void app_iface_dbus_on_name_aquired(GDBusConnection *con, const gchar* name, gpointer data)
{
	(void)data;
	(void)con;
	(void)name;
	logger_log_debug("APP_IFACE_DBUS - Connection to system bus established.");

	app_iface_set_ready();
}

static void app_iface_dbus_on_connection_lost(GDBusConnection *con, const gchar* name, gpointer data)
{
	(void)con;
	(void)name;
	(void)data;

	logger_log_error("Connection to the system bus lost. Shutting down.");
	app_iface_request_daemon_exit_due_to_errors(RESULT_DBUS_SYSTEM_BUS_ERROR);
}

static gboolean app_iface_dbus_on_challenge_request(Change_level *iface,GDBusMethodInvocation *invocation,
		const gchar *arg_ecu_id)
{
	const challenge_t *challenge;
	GVariant *challenge_ser;

	//using the unmodified ecu id for creating the challenge
	challenge=app_iface_create_new_challenge(arg_ecu_id);
	challenge_ser=g_variant_new_from_data((const GVariantType *)"ay",challenge,sizeof(challenge_t),TRUE,NULL, NULL);

	change_level_complete_challenge(iface,invocation,challenge_ser);

	return TRUE;
}

static gboolean app_iface_dbus_on_response_request(Change_level *iface,GDBusMethodInvocation *invocation,
		GVariant *arg_response)
{
	const challenge_response_t *response;
	daemon_level_state_t level_state;

	level_state=app_iface_get_daemon_level_state();

	//ensure we are not in the middle of request
	if (level_state==DAEMON_LEVEL_RECOVERY_ONGOING)
	{
		change_level_complete_response(iface,invocation,RESULT_DAEMON_BUSY_LEVEL_RECOVERY);
		return TRUE;
	}

	if (level_state==DAEMON_LEVEL_CHANGE_ONGOING)
	{
		change_level_complete_response(iface,invocation,RESULT_DAEMON_BUSY_LEVEL_CHANGE);
		return TRUE;
	}
	if(level_state==DAEMON_LEVEL_RECOVERY_ONSTARTUP_NEEDED)
	{
		change_level_complete_response(iface,invocation,RESULT_DAEMON_LEVEL_RECOVERY_ONSTARTUP_NEEDED);
		return TRUE;
	}

	response=g_variant_get_data(arg_response);

	if (g_variant_get_size(arg_response)!=sizeof(challenge_response_t))
	{
		logger_log_error("APP_IFACE_DBUS - Invalid response received. Canceling the request.");
		change_level_complete_response(iface,invocation, RESULT_CHALLENGE_INVALID_RESPONSE);
		return TRUE;
	}

	//pending_request_innvocation should be NULL (ensured by the FSM)
	pending_request_invocation=invocation;
	app_iface_request_level_change(response, app_iface_dbus_level_change_request_callback);

	return TRUE;
}

static gboolean app_iface_dbus_on_lock_device_request(Change_level *iface,GDBusMethodInvocation *invocation)
{
	daemon_level_state_t level_state;

	level_state=app_iface_get_daemon_level_state();

	//ensure we are not in the middle of a request
	if (level_state==DAEMON_LEVEL_RECOVERY_ONGOING)
	{
		change_level_complete_lock_device(iface,invocation,RESULT_DAEMON_BUSY_LEVEL_RECOVERY);
		return TRUE;
	}

	if (level_state==DAEMON_LEVEL_CHANGE_ONGOING)
	{
		change_level_complete_lock_device(iface,invocation,RESULT_DAEMON_BUSY_LEVEL_CHANGE);
		return TRUE;
	}

	//pending_request_innvocation should be NULL (ensured by the FSM)
	pending_request_invocation=invocation;

	app_iface_request_lock_device(app_iface_dbus_on_lock_device_request_callback);

	return TRUE;
}

static void app_iface_dbus_on_lock_device_request_callback(error_code_t result, bool did_level_change)
{
	(void)did_level_change;
	if (pending_request_invocation==NULL)
		return;

	change_level_complete_lock_device(change_level_iface,pending_request_invocation,result);
	logger_log_debug("APP_IFACE_DBUS - Finalizing lock_device call(Error Code: %d).",result);
	pending_request_invocation=NULL;
}

static void app_iface_dbus_level_change_request_callback(error_code_t result, bool did_level_change)
{
	(void)did_level_change;
	if (pending_request_invocation==NULL)
		return;

	change_level_complete_response(change_level_iface,pending_request_invocation,result);
	logger_log_debug("APP_IFACE_DBUS - Finalizing response call(Error Code: %d).",result);
	pending_request_invocation=NULL;
}
//--------------------------------------------------------------------------------------------------------------------
